import LicenseSpring

import AppKit
import SwiftUI

struct InstallationFilesTable: View {
    let values: [RandomIdentifiable<InstallationFile>]
    @Binding var selection: Int?
    
    var body: some View {
        Representable(content: .installationFiles(values)) { selection = $0 }
    }
}

struct LicenseFeaturesTable: View {
    let values: [RandomIdentifiable<LicenseFeature>]
    @Binding var selection: Int?
    
    var body: some View {
        Representable(content: .licenseFeatures(values)) { selection = $0 }
    }
}

struct CustomFieldsTable: View {
    let values: [RandomIdentifiable<CustomField>]
    @Binding var selection: Int?
    
    var body: some View {
        Representable(content: .customFields(values)) { selection = $0 }
    }
}

private enum TableContent {
    case installationFiles([RandomIdentifiable<InstallationFile>])
    case licenseFeatures([RandomIdentifiable<LicenseFeature>])
    case customFields([RandomIdentifiable<CustomField>])
}

private struct Representable: NSViewRepresentable {
    let content: TableContent
    let selectionChanged: (Int?) -> Void
    
    func makeNSView(context: Context) -> NSView {
        let view: NSView!
        switch content {
        case .installationFiles:
            view = context.coordinator.installationFilesView
        case .licenseFeatures:
            view = context.coordinator.licenseFeaturesView
        case .customFields:
            view = context.coordinator.customFieldsView
        }
        
        updateNSView(view, context: context)
        
        return view
    }
    
    func updateNSView(_ nsView: NSView, context: Context) {
        switch content {
        case .installationFiles(let values):
            context.coordinator.update(installationFiles: values.map(\.value))
        case .licenseFeatures(let values):
            context.coordinator.update(licenseFeatures: values.map(\.value))
        case .customFields(let values):
            context.coordinator.update(customFields: values.map(\.value))
        }
    }
    
    func makeCoordinator() -> TableBridgeViewController {
        let vc = TableBridgeViewController(nibName: "TableBridge", bundle: nil)
        vc.selectionChanged = { selection in
            Task{ @MainActor in
                selectionChanged(selection)
            }
        }
        vc.loadView()
        return vc
    }
}

private class TableBridgeViewController: NSViewController, NSTableViewDataSource {
    @IBOutlet weak var installationFilesView: NSView!
    @IBOutlet weak var installationFilesContent: NSArrayController!
    @IBOutlet weak var licenseFeaturesView: NSView!
    @IBOutlet weak var licenseFeaturesContent: NSArrayController!
    @IBOutlet weak var customFieldsView: NSView!
    @IBOutlet weak var customFieldsContent: NSArrayController!
    
    private var subscriptions: [NSKeyValueObservation] = []
    
    var selectionChanged: (@MainActor @Sendable (Int?) -> Void)?

    override func loadView() {
        super.loadView()
        
        subscriptions.append(
            installationFilesContent.observe(\.selectionIndex) { [weak self] array, _ in
                let index = array.selectionIndex != NSNotFound ? array.selectionIndex : nil
                Task { @MainActor in
                    self?.selectionChanged?(index)
                }
            }
        )
        subscriptions.append(
            licenseFeaturesContent.observe(\.selectionIndex) { [weak self] array, _ in
                let index = array.selectionIndex != NSNotFound ? array.selectionIndex : nil
                Task { @MainActor in
                    self?.selectionChanged?(index)
                }
            }
        )

        subscriptions.append(
            customFieldsContent.observe(\.selectionIndex) { [weak self] array, _ in
                let index = array.selectionIndex != NSNotFound ? array.selectionIndex : nil
                Task { @MainActor in
                    self?.selectionChanged?(index)
                }
            }
        )
        
        installationFilesContent.selectsInsertedObjects = false
        licenseFeaturesContent.selectsInsertedObjects = false
        customFieldsContent.selectsInsertedObjects = false
    }
    
    func update(installationFiles: [InstallationFile]) {
        installationFilesContent.content = installationFiles
    }
    
    func update(licenseFeatures: [LicenseFeature]) {
        licenseFeaturesContent.content = licenseFeatures
    }
    
    func update(customFields: [CustomField]) {
        customFieldsContent.content = customFields
    }
}

extension LicenseFeature {
    @objc
    var featureTypeString: String { featureType.description }
}

struct TableBridge_Previews: PreviewProvider {
    static var previews: some View {
        VStack {
            InstallationFilesTable(values: [], selection: .constant(nil))
            LicenseFeaturesTable(values: [], selection: .constant(nil))
            CustomFieldsTable(values: [], selection: .constant(nil))
        }
    }
}
